<?php
require_once 'config.php';
requireLogin();

// Get user's department
$department = getCurrentDepartment();

// Get statistics for dashboard
$stats = [];
if ($department === 'admin') {
    $stats['total_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients")->fetch_assoc()['count'];
    $stats['lead_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE current_department = 'lead'")->fetch_assoc()['count'];
    $stats['sales_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE current_department = 'sales'")->fetch_assoc()['count'];
    $stats['delivery_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE current_department = 'delivery'")->fetch_assoc()['count'];
    $stats['project_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE current_department = 'project'")->fetch_assoc()['count'];
    $stats['upfollow_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE current_department = 'upfollow'")->fetch_assoc()['count'];
    $stats['completed_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE status = 'completed'")->fetch_assoc()['count'];
    $stats['paid_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE package = 'paid'")->fetch_assoc()['count'];
    $stats['today_clients'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE DATE(created_at) = CURDATE()")->fetch_assoc()['count'];
} else {
    $stats['my_department'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE current_department = '$department'")->fetch_assoc()['count'];
    $stats['total_assigned'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE assigned_to = '{$_SESSION['user_id']}'")->fetch_assoc()['count'];
    $stats['pending_assigned'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE assigned_to = '{$_SESSION['user_id']}' AND status = 'pending'")->fetch_assoc()['count'];
    $stats['completed_assigned'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE assigned_to = '{$_SESSION['user_id']}' AND status = 'completed'")->fetch_assoc()['count'];
    $stats['today_assigned'] = $conn->query("SELECT COUNT(*) as count FROM clients WHERE assigned_to = '{$_SESSION['user_id']}' AND DATE(created_at) = CURDATE()")->fetch_assoc()['count'];
}

// Get recent clients for user's department
if ($department === 'admin') {
    $recent_clients = $conn->query("SELECT c.*, u.full_name as assigned_to_name FROM clients c LEFT JOIN users u ON c.assigned_to = u.id ORDER BY updated_at DESC LIMIT 10");
} else {
    $stmt = $conn->prepare("SELECT c.*, u.full_name as assigned_to_name FROM clients c LEFT JOIN users u ON c.assigned_to = u.id WHERE c.current_department = ? ORDER BY updated_at DESC LIMIT 10");
    $stmt->bind_param("s", $department);
    $stmt->execute();
    $recent_clients = $stmt->get_result();
}

// Get activity log for user
$activity_log = [];
if ($department === 'admin') {
    $activity_log_result = $conn->query("
        SELECT ch.*, c.name as client_name, u.full_name as action_by_name 
        FROM client_history ch 
        JOIN clients c ON ch.client_id = c.id 
        JOIN users u ON ch.action_by = u.id 
        ORDER BY ch.action_date DESC 
        LIMIT 10
    ");
} else {
    $activity_stmt = $conn->prepare("
        SELECT ch.*, c.name as client_name, u.full_name as action_by_name 
        FROM client_history ch 
        JOIN clients c ON ch.client_id = c.id 
        JOIN users u ON ch.action_by = u.id 
        WHERE ch.action_by = ? 
        ORDER BY ch.action_date DESC 
        LIMIT 10
    ");
    $activity_stmt->bind_param("i", $_SESSION['user_id']);
    $activity_stmt->execute();
    $activity_log_result = $activity_stmt->get_result();
}

while($activity = $activity_log_result->fetch_assoc()) {
    $activity_log[] = $activity;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard | Client Management System</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .dashboard-summary {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .welcome-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            border-radius: 10px;
            display: flex;
            align-items: center;
            gap: 1.5rem;
        }
        
        .welcome-icon {
            font-size: 3rem;
            opacity: 0.9;
        }
        
        .welcome-text h2 {
            margin: 0 0 0.5rem 0;
            font-size: 1.5rem;
        }
        
        .welcome-text p {
            margin: 0;
            opacity: 0.9;
        }
        
        .quick-stats {
            background: white;
            padding: 2rem;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        }
        
        .quick-stats h3 {
            margin-bottom: 1.5rem;
            color: #333;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .stat-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 1rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid #eee;
        }
        
        .stat-row:last-child {
            margin-bottom: 0;
            padding-bottom: 0;
            border-bottom: none;
        }
        
        .stat-label {
            color: #666;
        }
        
        .stat-value {
            font-weight: bold;
            color: #333;
        }
        
        .activity-log {
            background: white;
            padding: 2rem;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
            margin-top: 2rem;
        }
        
        .activity-item {
            padding: 1rem;
            border-bottom: 1px solid #eee;
            display: flex;
            align-items: flex-start;
            gap: 1rem;
        }
        
        .activity-item:last-child {
            border-bottom: none;
        }
        
        .activity-icon {
            width: 40px;
            height: 40px;
            background: #f8f9fa;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #667eea;
        }
        
        .activity-content {
            flex: 1;
        }
        
        .activity-client {
            font-weight: bold;
            color: #333;
            margin-bottom: 0.25rem;
        }
        
        .activity-action {
            color: #666;
            margin-bottom: 0.25rem;
        }
        
        .activity-meta {
            font-size: 0.875rem;
            color: #999;
        }
        
        .department-icon {
            width: 50px;
            height: 50px;
            background: #f8f9fa;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: #667eea;
            margin-bottom: 1rem;
        }
        
        .department-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .department-stat {
            background: white;
            padding: 1.5rem;
            border-radius: 10px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.05);
            text-align: center;
        }
        
        .empty-state {
            text-align: center;
            padding: 3rem;
            color: #666;
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: #ddd;
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="logo">
            <i class="fas fa-briefcase"></i>
            <span>Client Management System</span>
        </div>
        <div class="user-info">
            <span>Welcome, <?php echo htmlspecialchars($_SESSION['full_name']); ?> (<?php echo ucfirst($_SESSION['department']); ?>)</span>
            <a href="logout.php" class="btn-logout">
                <i class="fas fa-sign-out-alt"></i> Logout
            </a>
        </div>
    </div>

    <div class="sidebar">
        <ul class="nav-links">
            <li><a href="dashboard.php" class="active"><i class="fas fa-home"></i> Dashboard</a></li>
            <?php if ($department === 'lead' || $department === 'admin'): ?>
            <li><a href="lead_dept.php"><i class="fas fa-user-plus"></i> Lead Department</a></li>
            <?php endif; ?>
            <?php if ($department === 'sales' || $department === 'admin'): ?>
            <li><a href="sales_dept.php"><i class="fas fa-chart-line"></i> Sales Department</a></li>
            <?php endif; ?>
            <?php if ($department === 'delivery' || $department === 'admin'): ?>
            <li><a href="delivery_dept.php"><i class="fas fa-shipping-fast"></i> Delivery Department</a></li>
            <?php endif; ?>
            <?php if ($department === 'project' || $department === 'admin'): ?>
            <li><a href="project_dept.php"><i class="fas fa-project-diagram"></i> Project Management</a></li>
            <?php endif; ?>
            <?php if ($department === 'upfollow' || $department === 'admin'): ?>
            <li><a href="upfollow_dept.php"><i class="fas fa-chart-bar"></i> Upscale Department</a></li>
            <?php endif; ?>
            <?php if ($department === 'admin'): ?>
            <li><a href="users.php"><i class="fas fa-users"></i> User Management</a></li>
            <li><a href="reports.php"><i class="fas fa-chart-pie"></i> Reports</a></li>
            <?php endif; ?>
        </ul>
    </div>

    <div class="main-content">
        <div class="dashboard-summary">
            <div class="welcome-card">
                <div class="welcome-icon">
                    <i class="fas fa-user-tie"></i>
                </div>
                <div class="welcome-text">
                    <h2>Welcome back, <?php echo htmlspecialchars($_SESSION['full_name']); ?>!</h2>
                    <p><?php echo ucfirst($department); ?> Department</p>
                    <p><small><?php echo date('l, F j, Y'); ?></small></p>
                </div>
            </div>
            
            <div class="quick-stats">
                <h3><i class="fas fa-chart-bar"></i> Quick Stats</h3>
                <?php if ($department === 'admin'): ?>
                    <div class="stat-row">
                        <span class="stat-label">Total Clients</span>
                        <span class="stat-value"><?php echo $stats['total_clients']; ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Completed Projects</span>
                        <span class="stat-value"><?php echo $stats['completed_clients']; ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Paid Packages</span>
                        <span class="stat-value"><?php echo $stats['paid_clients']; ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Today's Clients</span>
                        <span class="stat-value"><?php echo $stats['today_clients']; ?></span>
                    </div>
                <?php else: ?>
                    <div class="stat-row">
                        <span class="stat-label">Clients in Department</span>
                        <span class="stat-value"><?php echo $stats['my_department']; ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Assigned to Me</span>
                        <span class="stat-value"><?php echo $stats['total_assigned']; ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Pending Tasks</span>
                        <span class="stat-value"><?php echo $stats['pending_assigned']; ?></span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">Today's Assignments</span>
                        <span class="stat-value"><?php echo $stats['today_assigned']; ?></span>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if ($department === 'admin'): ?>
        <div class="department-stats">
            <div class="department-stat">
                <div class="department-icon">
                    <i class="fas fa-user-plus"></i>
                </div>
                <h3>Lead Department</h3>
                <div class="count"><?php echo $stats['lead_clients']; ?></div>
            </div>
            <div class="department-stat">
                <div class="department-icon">
                    <i class="fas fa-chart-line"></i>
                </div>
                <h3>Sales Department</h3>
                <div class="count"><?php echo $stats['sales_clients']; ?></div>
            </div>
            <div class="department-stat">
                <div class="department-icon">
                    <i class="fas fa-shipping-fast"></i>
                </div>
                <h3>Delivery Department</h3>
                <div class="count"><?php echo $stats['delivery_clients']; ?></div>
            </div>
            <div class="department-stat">
                <div class="department-icon">
                    <i class="fas fa-project-diagram"></i>
                </div>
                <h3>Project Department</h3>
                <div class="count"><?php echo $stats['project_clients']; ?></div>
            </div>
            <div class="department-stat">
                <div class="department-icon">
                    <i class="fas fa-chart-bar"></i>
                </div>
                <h3>Upscale Department</h3>
                <div class="count"><?php echo $stats['upfollow_clients']; ?></div>
            </div>
        </div>
        <?php endif; ?>

        <div class="table-container">
            <h2><i class="fas fa-users"></i> Recent Clients</h2>
            <?php if ($recent_clients->num_rows == 0): ?>
                <div class="empty-state">
                    <i class="fas fa-users"></i>
                    <h3>No clients found</h3>
                    <p>No recent clients in <?php echo $department === 'admin' ? 'the system' : 'your department'; ?></p>
                </div>
            <?php else: ?>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Phone</th>
                            <th>Package</th>
                            <th>Department</th>
                            <th>Status</th>
                            <th>Assigned To</th>
                            <th>Last Updated</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while($client = $recent_clients->fetch_assoc()): ?>
                        <tr>
                            <td>#<?php echo str_pad($client['id'], 5, '0', STR_PAD_LEFT); ?></td>
                            <td><strong><?php echo htmlspecialchars($client['name']); ?></strong></td>
                            <td><?php echo htmlspecialchars($client['phone']); ?></td>
                            <td>
                                <span class="status-badge <?php echo $client['package'] === 'paid' ? 'status-completed' : 'status-pending'; ?>">
                                    <?php echo ucfirst($client['package']); ?>
                                </span>
                            </td>
                            <td>
                                <span class="status-badge <?php echo $client['current_department'] === 'lead' ? 'status-pending' : ($client['current_department'] === 'completed' ? 'status-completed' : 'status-in-progress'); ?>">
                                    <?php echo ucfirst($client['current_department']); ?>
                                </span>
                            </td>
                            <td>
                                <span class="status-badge status-<?php echo str_replace('_', '-', $client['status']); ?>">
                                    <?php echo ucwords(str_replace('_', ' ', $client['status'])); ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($client['assigned_to_name'] ?? 'Not Assigned'); ?></td>
                            <td title="<?php echo date('Y-m-d H:i:s', strtotime($client['updated_at'])); ?>">
                                <?php echo date('M d, H:i', strtotime($client['updated_at'])); ?>
                            </td>
                            <td>
                                <div class="btn-group">
                                    <button class="btn btn-primary btn-sm" onclick="viewClient(<?php echo $client['id']; ?>)">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                    <?php if ($department === $client['current_department'] || $department === 'admin'): ?>
                                    <a href="<?php echo $client['current_department']; ?>_dept.php?edit=<?php echo $client['id']; ?>" class="btn btn-secondary btn-sm">
                                        <i class="fas fa-edit"></i> Edit
                                    </a>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>

        <div class="activity-log">
            <h2><i class="fas fa-history"></i> Recent Activity</h2>
            <?php if (empty($activity_log)): ?>
                <div class="empty-state">
                    <i class="fas fa-history"></i>
                    <h3>No recent activity</h3>
                    <p>Your activity log will appear here</p>
                </div>
            <?php else: ?>
                <div class="activity-list">
                    <?php foreach($activity_log as $activity): 
                        $action_date = new DateTime($activity['action_date']);
                        $now = new DateTime();
                        $interval = $now->diff($action_date);
                        
                        if ($interval->d > 0) {
                            $time_ago = $interval->d . ' days ago';
                        } elseif ($interval->h > 0) {
                            $time_ago = $interval->h . ' hours ago';
                        } elseif ($interval->i > 0) {
                            $time_ago = $interval->i . ' minutes ago';
                        } else {
                            $time_ago = 'Just now';
                        }
                    ?>
                    <div class="activity-item">
                        <div class="activity-icon">
                            <?php if (strpos($activity['notes'], 'created') !== false): ?>
                                <i class="fas fa-plus-circle"></i>
                            <?php elseif (strpos($activity['notes'], 'updated') !== false): ?>
                                <i class="fas fa-edit"></i>
                            <?php elseif (strpos($activity['notes'], 'referred') !== false || strpos($activity['notes'], 'transferred') !== false): ?>
                                <i class="fas fa-exchange-alt"></i>
                            <?php elseif (strpos($activity['notes'], 'sent back') !== false): ?>
                                <i class="fas fa-arrow-left"></i>
                            <?php else: ?>
                                <i class="fas fa-info-circle"></i>
                            <?php endif; ?>
                        </div>
                        <div class="activity-content">
                            <div class="activity-client"><?php echo htmlspecialchars($activity['client_name']); ?></div>
                            <div class="activity-action"><?php echo htmlspecialchars($activity['notes']); ?></div>
                            <div class="activity-meta">
                                <?php echo htmlspecialchars($activity['action_by_name']); ?> • 
                                <?php echo $time_ago; ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Client View Modal -->
    <div class="modal" id="clientModal">
        <div class="modal-content" style="max-width: 800px;">
            <div class="modal-header">
                <h3>Client Details</h3>
                <button class="btn-close" onclick="closeModal()">&times;</button>
            </div>
            <div id="clientDetails">
                <div class="loading" style="text-align: center; padding: 2rem;">
                    <i class="fas fa-spinner fa-spin"></i> Loading client details...
                </div>
            </div>
        </div>
    </div>

    <script>
        function viewClient(clientId) {
            const modal = document.getElementById('clientModal');
            const details = document.getElementById('clientDetails');
            
            details.innerHTML = `
                <div style="text-align: center; padding: 2rem;">
                    <i class="fas fa-spinner fa-spin"></i> Loading client details...
                </div>
            `;
            
            modal.classList.add('active');
            
            fetch(`api/get_client.php?id=${clientId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        details.innerHTML = `
                            <div class="error" style="padding: 2rem; text-align: center; color: #dc3545;">
                                <i class="fas fa-exclamation-circle"></i>
                                <h4>Error Loading Client</h4>
                                <p>${data.error}</p>
                            </div>
                        `;
                        return;
                    }
                    
                    // Format dates
                    const createdDate = new Date(data.created_at);
                    const updatedDate = new Date(data.updated_at);
                    
                    details.innerHTML = `
                        <div class="client-details-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 1.5rem;">
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Client ID</strong>
                                <p>#${data.id.toString().padStart(5, '0')}</p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Name</strong>
                                <p>${escapeHtml(data.name)}</p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Phone</strong>
                                <p>${escapeHtml(data.phone)}</p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Package</strong>
                                <p><span class="status-badge ${data.package === 'paid' ? 'status-completed' : 'status-pending'}" style="display: inline-block; padding: 0.25rem 0.75rem; border-radius: 20px; font-size: 0.875rem; background: ${data.package === 'paid' ? '#d4edda' : '#fff3cd'}; color: ${data.package === 'paid' ? '#155724' : '#856404'};">${data.package}</span></p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Country</strong>
                                <p>${escapeHtml(data.country || 'N/A')}</p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Current Department</strong>
                                <p>${data.current_department}</p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Status</strong>
                                <p><span class="status-badge status-${data.status.replace('_', '-')}" style="display: inline-block; padding: 0.25rem 0.75rem; border-radius: 20px; font-size: 0.875rem; background: ${data.status === 'completed' ? '#d4edda' : (data.status === 'in_progress' ? '#d1ecf1' : '#fff3cd')}; color: ${data.status === 'completed' ? '#155724' : (data.status === 'in_progress' ? '#0c5460' : '#856404')};">${data.status.replace('_', ' ')}</span></p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Payment Status</strong>
                                <p><span class="status-badge ${data.payment_status === 'collected' ? 'status-completed' : 'status-pending'}" style="display: inline-block; padding: 0.25rem 0.75rem; border-radius: 20px; font-size: 0.875rem; background: ${data.payment_status === 'collected' ? '#d4edda' : '#fff3cd'}; color: ${data.payment_status === 'collected' ? '#155724' : '#856404'};">${data.payment_status}</span></p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Created</strong>
                                <p>${createdDate.toLocaleDateString()} ${createdDate.toLocaleTimeString()}</p>
                            </div>
                            <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px;">
                                <strong>Last Updated</strong>
                                <p>${updatedDate.toLocaleDateString()} ${updatedDate.toLocaleTimeString()}</p>
                            </div>
                        </div>
                        
                        <div class="detail-item" style="background: #f8f9fa; padding: 1rem; border-radius: 5px; margin-top: 1rem;">
                            <strong>Note</strong>
                            <p style="white-space: pre-wrap; margin-top: 0.5rem;">${escapeHtml(data.note || 'No notes available')}</p>
                        </div>
                        
                        <div style="margin-top: 1.5rem; text-align: center;">
                            <a href="${data.current_department}_dept.php" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: 0.5rem; padding: 0.5rem 1rem; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none; border-radius: 5px; text-decoration: none;">
                                <i class="fas fa-external-link-alt"></i> Go to ${data.current_department} Department
                            </a>
                        </div>
                    `;
                })
                .catch(error => {
                    console.error('Error:', error);
                    details.innerHTML = `
                        <div class="error" style="padding: 2rem; text-align: center; color: #dc3545;">
                            <i class="fas fa-exclamation-circle"></i>
                            <h4>Failed to load client details</h4>
                            <p>Please try again or check your connection.</p>
                        </div>
                    `;
                });
        }

        function closeModal() {
            document.getElementById('clientModal').classList.remove('active');
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('clientModal');
            if (event.target === modal) {
                closeModal();
            }
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // Auto-refresh dashboard every 60 seconds
        setTimeout(() => {
            location.reload();
        }, 60000);
    </script>
</body>
</html>